package com.hxstrive.service_demo.controller;

import com.hxstrive.service_demo.dto.CommonReturn;
import com.hxstrive.service_demo.entity.User;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * Restful 风格的用户控制器
 * @author hxstrive.com
 */
@Slf4j
@RestController
@RequestMapping("/user/restful")
public class UserRestfulController {
    private final static List<User> USERS = new ArrayList<>();

    static {
        USERS.add(User.builder().id(1L).name("Tom").age(20).build());
        USERS.add(User.builder().id(2L).name("Helen").age(30).build());
        USERS.add(User.builder().id(3L).name("Bill").age(40).build());
    }

    @Value("${spring.application.name}")
    private String appName;

    @Value("${server.port}")
    private String appPort;


    // GET 请求，获取所有用户信息
    @GetMapping("/getAllUsers")
    public CommonReturn<List<User>> getAllUsers() {
        log.info("getAllUsers()");
        return CommonReturn.success(USERS).ext(appName, appPort);
    }

    // GET 请求，根据用户 ID 获取用户信息
    @GetMapping("/{id}")
    public CommonReturn<User> getUserById(@PathVariable Long id) {
        log.info("getUserById() id={}", id);
        return USERS.stream()
                .filter(user -> user.getId().equals(id))
                .findFirst()
                .map((u) -> CommonReturn.success(u).ext(appName, appPort))
                .orElse(CommonReturn.fail("用户不存在"));
    }

    // POST 请求，创建新用户
    @PostMapping("/createUser")
    public CommonReturn<User> createUser(@RequestBody User user) {
        log.info("createUser() user={}", user);
        USERS.add(user);
        return CommonReturn.success(user).ext(appName, appPort);
    }

    // PUT 请求，更新用户信息
    @PutMapping("/{id}")
    public CommonReturn<User> updateUser(@PathVariable Long id, @RequestBody User updatedUser) {
        log.info("updateUser() id={}, updateUser={}", id, updatedUser);
        for (int i = 0; i < USERS.size(); i++) {
            if (USERS.get(i).getId().equals(id)) {
                USERS.set(i, updatedUser);
                return CommonReturn.success(updatedUser).ext(appName, appPort);
            }
        }
        return CommonReturn.fail("更新用户信息失败");
    }

    // DELETE 请求，删除用户
    @DeleteMapping("/{id}")
    public CommonReturn<String> deleteUser(@PathVariable Long id) {
        log.info("deleteUser() id={}", id);
        USERS.removeIf(user -> user.getId().equals(id));
        return CommonReturn.success("删除成功");
    }
}
