package com.hxstrive.hystrix_demo.controller;

import com.hxstrive.hystrix_demo.dto.CommonReturn;
import com.hxstrive.hystrix_demo.entity.User;
import com.hxstrive.hystrix_demo.service.UserService;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixCommand;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixProperty;
import com.netflix.hystrix.contrib.javanica.conf.HystrixPropertiesManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * Hystrix 熔断
 * @author hxstrive.com
 */
@RestController
@RequestMapping("/demo3")
public class Demo3Controller {

    @Autowired
    private UserService userService;

    @GetMapping("/getUserById")
    @HystrixCommand(fallbackMethod = "fallback",
        commandProperties = {
                // 配置熔断器的参数
                // 在一个滚动时间窗口内，请求的数量达到这个阈值时，Hystrix 才会开始去判断是否要开启熔断器
                @HystrixProperty(name = "circuitBreaker.requestVolumeThreshold", value = "5"),
                // 属性定义了在满足 circuitBreaker.requestVolumeThreshold（即请求量达到一定阈值）的前提下，
                // 失败请求数量占总请求数量的百分比达到多少时，熔断器就会开启
                @HystrixProperty(name = "circuitBreaker.errorThresholdPercentage", value = "50"),
                // 当熔断器开启之后，经过指定的这个时间（单位是毫秒），熔断器会进入半开状态。
                @HystrixProperty(name = "circuitBreaker.sleepWindowInMilliseconds", value = "5000")
    })
    public CommonReturn<User> getUserById(@RequestParam Long id) {
        // 更多参数可见 HystrixPropertiesManager 类
        //HystrixPropertiesManager.CIRCUIT_BREAKER_ENABLED;
        return userService.getUserById(id);
    }

    /**
     * 降级方法，当服务熔断或出现异常时被调用
     */
    private CommonReturn<User> fallback(Long id) {
        return CommonReturn.fail("网络出现问题，调用 fallback 方法，id=" + id);
    }

}
