package com.hxstrive.hystrix_demo.controller;

import com.hxstrive.hystrix_demo.dto.CommonReturn;
import com.hxstrive.hystrix_demo.entity.User;
import com.hxstrive.hystrix_demo.service.UserService;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixCommand;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixProperty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;

/**
 * Hystrix 信号量隔离
 * @author hxstrive.com
 */
@RestController
@RequestMapping("/demo6")
public class Demo6Controller {

    @Autowired
    private UserService userService;

    @GetMapping("/getUserById")
    // 使用HystrixCommand注解来标记这个方法受Hystrix管控，并配置信号量隔离相关属性
    @HystrixCommand(fallbackMethod = "fallback",
            commandProperties = {
                    // 设置隔离策略为信号量隔离
                    @HystrixProperty(name = "execution.isolation.strategy", value = "SEMAPHORE"),
                    // 设置信号量最大并发请求数量，这里设为3
                    @HystrixProperty(name = "execution.isolation.semaphore.maxConcurrentRequests", value = "3")
    })
    public CommonReturn<User> getUserById(HttpServletResponse response, @RequestParam Long id) {
        try {
            Thread.sleep(400);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return userService.getUserById(id);
    }

    /**
     * 当服务调用失败时，调用此方法
     */
    private CommonReturn<User> fallback(HttpServletResponse response, Long id) {
        response.setStatus(500);
        return CommonReturn.fail("网络出现问题，调用 fallback 方法，id=" + id);
    }

}
