package com.hxstrive.hystrix_demo.controller;

import com.hxstrive.hystrix_demo.dto.CommonReturn;
import com.hxstrive.hystrix_demo.entity.User;
import com.hxstrive.hystrix_demo.service.UserService;
import com.netflix.hystrix.contrib.javanica.annotation.DefaultProperties;
import com.netflix.hystrix.contrib.javanica.annotation.HystrixCommand;
import com.netflix.hystrix.contrib.javanica.cache.annotation.CacheKey;
import com.netflix.hystrix.contrib.javanica.cache.annotation.CacheRemove;
import com.netflix.hystrix.contrib.javanica.cache.annotation.CacheResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.UUID;

/**
 * Hystrix 缓存
 * @author hxstrive.com
 */
@RestController
@RequestMapping("/demo8")
@DefaultProperties
public class Demo8Controller {

    @Autowired
    private UserService userService;

    @GetMapping("/getUserById")
    // 使用 @CacheResult 注解开启缓存，并指定缓存键生成方法
    @CacheResult(cacheKeyMethod = "generateCacheKey")
    @HystrixCommand(fallbackMethod = "getUserByIdFallback", commandKey = "user_key")
    public CommonReturn<User> getUserById(@CacheKey @RequestParam Long id) {
        return userService.getUserById(id);
    }

    public CommonReturn<User> getUserByIdFallback(Long id) {
        return CommonReturn.fail("触发降级，返回 getUserByIdFallback");
    }

    // 生成缓存键的方法
    private String generateCacheKey(@CacheKey Long id) {
        System.out.println("generateCacheKey() id=" + id);
        return "myCacheKey:" + id;
    }

    @GetMapping("/updateUser")
    // 使用 @CacheRemove 注解清除缓存，指定要清除缓存的命令键和缓存键生成方法
    @CacheRemove(commandKey = "user_key", cacheKeyMethod = "generateCacheKey")
    @HystrixCommand(fallbackMethod = "updateUserFallback")
    public CommonReturn<User> updateUser(@RequestParam Long id) {
        CommonReturn<User> commonReturn = userService.getUserById(id);
        User user = commonReturn.getData();
        user.setName(UUID.randomUUID().toString());
        return userService.updateUser(id, user);
    }

    public CommonReturn<String> updateUserFallback(Long id) {
        return CommonReturn.fail("触发降级，返回 updateUserFallback");
    }

}
