package com.hxstrive.springcloud.user.controller;

import com.hxstrive.springcloud.user.entity.User;
import com.hxstrive.springcloud.user.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import java.util.List;

@RestController
@RequestMapping("/api/users")
public class UserController {

    /**
     * 节点名称，启动服务时通过参数传递
     */
    @Value("${nodeName:unknown}")
    private String nodeName;

    private final UserService userService;

    // 注入UserService实现类
    @Autowired
    public UserController(UserService userService) {
        this.userService = userService;
    }

    /**
     * 新增用户 - POST /api/users
     */
    @PostMapping
    public ResponseEntity<String> createUser(@RequestBody User user) {
        userService.save(user);
        // 返回201 Created状态码，Location头指向新资源URL
        return ResponseEntity
                .status(HttpStatus.CREATED)
                .header("Location", "/api/users/" + user.getId())
                .body("[" + nodeName + "] 创建用户成功");
    }

    /**
     * 根据ID查询用户 - GET /api/users/{id}
     */
    @GetMapping("/{id}")
    public ResponseEntity<String> getUserById(@PathVariable Integer id) {
        User user = userService.findById(id);
        if (user != null) {
            return ResponseEntity.ok("[" + nodeName + "] " + user); // 200 OK
        } else {
            return ResponseEntity.notFound().build(); // 404 Not Found
        }
    }

    /**
     * 查询所有用户 - GET /api/users
     */
    @GetMapping
    public ResponseEntity<String> getAllUsers() {
        List<User> users = userService.findAll();
        StringBuilder builder = new StringBuilder();
        for (User user : users) {
            builder.append("\n").append(user.toString());
        }
        return ResponseEntity.ok("[" + nodeName + "]" + builder); // 200 OK
    }

    /**
     * 更新用户 - PUT /api/users/{id}
     */
    @PutMapping("/{id}")
    public ResponseEntity<String> updateUser(
            @PathVariable Integer id,
            @RequestBody User user) {
        // 确保路径ID与请求体ID一致
        if (!id.equals(user.getId())) {
            return ResponseEntity.badRequest().build(); // 400 Bad Request
        }
        
        User existingUser = userService.findById(id);
        if (existingUser != null) {
            userService.update(user);
            return ResponseEntity.ok("[" + nodeName + "] 更新用户成功");
        } else {
            return ResponseEntity.notFound().build(); // 404 Not Found
        }
    }

    /**
     * 删除用户 - DELETE /api/users/{id}
     */
    @DeleteMapping("/{id}")
    public ResponseEntity<String> deleteUser(@PathVariable Integer id) {
        User existingUser = userService.findById(id);
        if (existingUser != null) {
            userService.delete(id);
            return ResponseEntity.ok("[" + nodeName + "] 删除用户成功");
        } else {
            return ResponseEntity.notFound().build(); // 404 Not Found
        }
    }
}